<?php
declare(strict_types=1);

namespace WP_Rocket\ThirdParty\Plugins\SEO;

use WP_Rocket\Admin\Options_Data;
use WP_Rocket\Engine\Admin\Settings\Settings;
use WP_Rocket\Event_Management\Subscriber_Interface;

class Yoast implements Subscriber_Interface {
	/**
	 * Options Data instance
	 *
	 * @var Options_Data
	 */
	private $options;

	/**
	 * Constructor
	 *
	 * @param Options_Data $options Options_Data instance.
	 */
	public function __construct( Options_Data $options ) {
		$this->options = $options;
	}

	/**
	 * Array of events this subscriber listens to
	 *
	 * @return array
	 */
	public static function get_subscribed_events() {
		return [
			'rocket_sitemap_preload_options' => 'add_settings_field',
			'rocket_sitemap_preload_list'    => 'add_sitemap',
			'rocket_input_sanitize'          => [ 'sanitize_option', 10, 2 ],
			'rocket_first_install_options'   => 'add_option',
		];
	}

	/**
	 * Add Yoast SEO option to WP Rocket settings
	 *
	 * @since 2.8
	 *
	 * @param array $options WP Rocket settings array.
	 *
	 * @return array
	 */
	public function add_settings_field( $options ): array {
		if ( ! $this->is_sitemap_enabled() ) {
			return $options;
		}

		$options['yoast_xml_sitemap'] = [
			'type'              => 'checkbox',
			'container_class'   => [
				'wpr-field--children',
			],
			'label'             => __( 'Yoast SEO XML sitemap', 'rocket' ),
			// translators: %s = Name of the plugin.
			'description'       => sprintf( __( 'We automatically detected the sitemap generated by the %s plugin. You can check the option to preload it.', 'rocket' ), 'Yoast SEO' ),
			'parent'            => 'sitemap_preload',
			'section'           => 'preload_section',
			'page'              => 'preload',
			'default'           => 0,
			'sanitize_callback' => 'sanitize_checkbox',
		];

		return $options;
	}

	/**
	 * Add Yoast SEO sitemap URL to the sitemaps to preload
	 *
	 * @since 2.8
	 *
	 * @param array $sitemaps An array of sitemaps to preload.
	 *
	 * @return array
	 */
	public function add_sitemap( $sitemaps ): array {
		if ( ! $this->is_sitemap_enabled() ) {
			return $sitemaps;
		}

		if ( ! $this->options->get( 'yoast_xml_sitemap', 0 ) ) {
			return $sitemaps;
		}

		if ( ! class_exists( 'WPSEO_Sitemaps_Router' ) ) {
			return $sitemaps;
		}

		$sitemaps[] = \WPSEO_Sitemaps_Router::get_base_url( 'sitemap_index.xml' );

		return $sitemaps;
	}

	/**
	 * Sanitize Yoast SEO sitemap option value
	 *
	 * @since 2.8
	 *
	 * @param array    $input Array of values submitted from the form.
	 * @param Settings $settings Settings class instance.
	 *
	 * @return array
	 */
	public function sanitize_option( $input, Settings $settings ): array {
		if ( ! $this->is_sitemap_enabled() ) {
			return $input;
		}

		$input['yoast_xml_sitemap'] = $settings->sanitize_checkbox( $input, 'yoast_xml_sitemap' );

		return $input;
	}

	/**
	 * Add Yoast SEO sitemap option to WP Rocket default options
	 *
	 * @since 2.8
	 *
	 * @param array $options WP Rocket options array.
	 *
	 * @return array
	 */
	public function add_option( $options ): array {
		if ( ! $this->is_sitemap_enabled() ) {
			return $options;
		}

		$options['yoast_xml_sitemap'] = 0;

		return $options;
	}

	/**
	 * Checks if sitemap is enabled in Yoast SEO
	 *
	 * @since 3.11.1
	 *
	 * @return bool
	 */
	private function is_sitemap_enabled(): bool {
		static $enabled = null;

		if ( ! is_null( $enabled ) ) {
			return $enabled;
		}

		if ( ! rocket_has_constant( 'WPSEO_VERSION' ) ) {
			$enabled = false;

			return $enabled;
		}

		$yoast_seo_xml = get_option( 'wpseo_xml', [] ); // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals

		if ( version_compare( rocket_get_constant( 'WPSEO_VERSION', '' ), '7.0' ) >= 0 ) {
			$yoast_seo                         = get_option( 'wpseo', [] ); // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals
			$yoast_seo_xml['enablexmlsitemap'] = isset( $yoast_seo['enable_xml_sitemap'] ) && $yoast_seo['enable_xml_sitemap']; // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals
		}

		$enabled = (bool) $yoast_seo_xml['enablexmlsitemap'];

		return $enabled;
	}
}
